/*===========================================================================*/
/* SORENSEN SGA                                                              */
/* LabWindows/CVI Instrument Driver                                          */
/* Original Release:                                                         */
/*                                                                           */
/* Modification History: None                                                */
/*===========================================================================*/

#include <visa.h>
#include <userint.h>
#include <ansi_c.h>
#include <formatio.h>
#include <string.h>
#include "SNSGA.h"

#define SNSGA_REVISION   "V1.0.0.1, 3/2010, CVI 8.5.0"    /* Driver Revision */  
#define BUFFER_SIZE      256L                          /* File I/O buffer */

#define DEMO_IF			0
#define GPIB_IF			1
#define RS232_IF		2
#define ENET_IF			3

#define LOCK                                                      \
    if ((SNSGA_status = viLock(instrSession,					  \
                                  VI_EXCLUSIVE_LOCK,			  \
                                  THREAD_LOCK_TIMEOUT,			  \
                                  NULL,NULL)) < 0)                \
        return SNSGA_status;                                      \

#define UNLOCK                                                    \
{                                                                 \
    if((unlockStatus = viUnlock(instrSession)) < 0)               \
        return unlockStatus;                                      \
    return SNSGA_status;                                          \
}

/*****************************************************************************/
/*= INSTRUMENT-DEPENDENT STATUS/RANGE STRUCTURE  ============================*/
/*****************************************************************************/
/* SNSGA_stringValPair is used in the SNSGA_errorMessage function          */
/* SNSGA_statusDataRanges is used to track session dependent status & ranges*/
/*===========================================================================*/
typedef struct  SNSGA_stringValPair
{
   ViStatus stringVal;
   ViString stringName;
}  SNSGA_tStringValPair;
 

typedef struct SNSGA_statusDataRanges *SNSGA_instrRange;

static short int		user_if;  

/*****************************************************************************/
/*= UTILITY ROUTINE DECLARATIONS (Non-Exportable Functions) =================*/
/*****************************************************************************/
ViBoolean SNSGA_invalidViBooleanRange (ViBoolean val);
ViBoolean SNSGA_invalidViInt16Range   (ViInt16 val, ViInt16 min, ViInt16 max);
ViBoolean SNSGA_invalidViInt32Range   (ViInt32 val, ViInt32 min, ViInt32 max);
ViBoolean SNSGA_invalidViUInt8Range   (ViUInt8 val, ViUInt8 min, ViUInt8 max);
ViBoolean SNSGA_invalidViUInt16Range  (ViUInt16 val, ViUInt16 min, ViUInt16 max);
ViBoolean SNSGA_invalidViUInt32Range  (ViUInt32 val, ViUInt32 min, ViUInt32 max);
ViBoolean SNSGA_invalidViReal32Range  (ViReal32 val, ViReal32 min, ViReal32 max);
ViBoolean SNSGA_invalidViReal64Range  (ViReal64 val, ViReal64 min, ViReal64 max);
ViStatus  SNSGA_initCleanUp           (ViSession openRMSession,
                                          ViPSession openInstrSession,
                                          ViStatus currentStatus);

/*****************************************************************************/
/*====== USER-CALLABLE FUNCTIONS (Exportable Functions) =====================*/
/*****************************************************************************/

/*===========================================================================*/
/* Function: Initialize                                                      */
/* Purpose:  This function opens the instrument, queries the instrument      */
/*           for its ID, and initializes the instrument to a known state.    */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_init (ViRsrc     resourceName,
                                ViBoolean  IDQuery,
                                ViBoolean  resetDevice,
                                ViPSession instrSession)
{
    ViStatus	SNSGA_status,unlockStatus;
    ViSession	rmSession = 0;
    ViUInt32	retCnt = 0;
    ViInt16		i;
    ViByte		rdBuffer[BUFFER_SIZE];
    ViByte		tmpBuffer[10];

    /*- Open instrument session ---------------------------------------------*/
    user_if = GPIB_IF;
    if ((SNSGA_status = viOpenDefaultRM (&rmSession)) < 0)
    {
    	sprintf(rdBuffer,"viOpenDefaultRM - %08X",SNSGA_status);
		MessagePopup("SNSGA Error",rdBuffer);
        return SNSGA_status;
    }

    if ((SNSGA_status = viOpen (rmSession, resourceName, VI_NULL, VI_NULL, instrSession)) < 0)
    {
    	sprintf(rdBuffer,"viOpen - %08X",SNSGA_status);
		MessagePopup("SNSGA Error",rdBuffer);
        viClose (rmSession);
        return SNSGA_status;
    }
	viSetAttribute (*instrSession, VI_ATTR_TMO_VALUE, 7000);  // RBF 7 sec TMO

	/* set up serial port if selected */
	if (strstr (resourceName, "ASRL") != NULL)
	{
		user_if = RS232_IF;
		viSetAttribute (*instrSession, VI_ATTR_ASRL_BAUD, 19200);
		viSetAttribute (*instrSession, VI_ATTR_ASRL_DATA_BITS, 8);
		viSetAttribute (*instrSession, VI_ATTR_ASRL_PARITY, VI_ASRL_PAR_NONE);
		viSetAttribute (*instrSession, VI_ATTR_ASRL_STOP_BITS, VI_ASRL_STOP_ONE);
		viSetAttribute (*instrSession, VI_ATTR_ASRL_FLOW_CNTRL, VI_ASRL_FLOW_NONE);
		viSetAttribute (*instrSession, VI_ATTR_ASRL_END_OUT, VI_ASRL_END_TERMCHAR);
		viSetAttribute (*instrSession, VI_ATTR_TERMCHAR, '\r');
	}
	else if ((resourceName, "TCPIP") != NULL)
	{
		user_if = ENET_IF;
		/*
		viSetAttribute (*instrSession, VI_ATTR_TERMCHAR_EN, VI_TRUE);
		viSetAttribute (*instrSession, VI_ATTR_TERMCHAR, '\n');
		*/
      	// 08-14-06 RBF
      	
      	// NIMAX says these are all settable attributes and thier defaults for TCPIP
      	//VI_ATTR_TMO_VALUE = 2000
      	//VI_ATTR_MAX_QUEUE_LENGTH = 50
      	//VI_ATTR_SEND_END_EN = VI_TRUE
      	//VI_ATTR_TERMCHAR = 0x0A   // linefeed \n
      	//VI_ATTR_TERMCHAR_EN = VI_FALSE
      	//VI_ATTR_DMA_ALLOW_EN = VI_FALSE
      	//VI_ATTR_FILE_APPEND_EN = VI_FALSE
      	//VI_ATTR_TCPIP_KEEPALIVE = VI_FALSE
      	
      	
      	// These may be settable and these are thier defaults for TCPIP
      	//VI_ATTR_TCPIP_NODELAY = VI_TRUE
	    //VI_ATTR_IO_PROT = VI_PROT_NORMAL
      	//VI_ATTR_SUPPRESS_END_EN = VI_TRUE
      	//VI_ATTR_WR_BUF_OPER_MODE = VI_FLUSH_WHEN_FULL
    	//VI_ATTR_USER_DATA = 0x00000000

	  	// Terminator /r or  xOD or cr  or carriage return only 
	  	// no /n or x0A or nl or new line, just to be clear

      	// NIMAX says these are all settable attributes:
		viSetAttribute (*instrSession, VI_ATTR_TMO_VALUE, 8000);
		viSetAttribute (*instrSession, VI_ATTR_MAX_QUEUE_LENGTH, 50);
		viSetAttribute (*instrSession, VI_ATTR_SEND_END_EN, VI_TRUE);
		viSetAttribute (*instrSession, VI_ATTR_TERMCHAR_EN, VI_TRUE);
		viSetAttribute (*instrSession, VI_ATTR_DMA_ALLOW_EN, VI_FALSE);
		viSetAttribute (*instrSession, VI_ATTR_FILE_APPEND_EN, VI_FALSE);
		viSetAttribute (*instrSession, VI_ATTR_TCPIP_KEEPALIVE, VI_FALSE);

		viSetAttribute (*instrSession, VI_ATTR_TERMCHAR, '\r'); 
		
	    // Bonus attemtp to set other attributes not deemed settable by NIMAX but might help:
      	viSetAttribute (*instrSession, VI_ATTR_TCPIP_NODELAY, VI_TRUE );
	    viSetAttribute (*instrSession, VI_ATTR_IO_PROT, VI_PROT_NORMAL );
      	viSetAttribute (*instrSession, VI_ATTR_SUPPRESS_END_EN, VI_TRUE );
      	viSetAttribute (*instrSession, VI_ATTR_WR_BUF_OPER_MODE, VI_FLUSH_WHEN_FULL );
    	viSetAttribute (*instrSession, VI_ATTR_USER_DATA, 0x00000000);
		
	}
	
	if ((SNSGA_status = SNSGA_ConfigTerm (*instrSession,"0",1)) != VI_SUCCESS)//force visa layer and unit to "\r"                        
		
    /*- Identification Query ------------------------------------------------*/
    if (IDQuery)					  
    {
		if ((SNSGA_status = SNSGA_idQuery (*instrSession,rdBuffer)) < VI_SUCCESS)
            return SNSGA_initCleanUp (rmSession, instrSession, SNSGA_status);

		/* get and compare first eight bytes from *idn? response */
		for (i = 0; i < 8; ++i)
			tmpBuffer[i] = rdBuffer[i];
		tmpBuffer[8] = '\0';
		StringUpperCase (tmpBuffer);
		if (strcmp(tmpBuffer,"SORENSEN") != 0)
            return SNSGA_initCleanUp (rmSession, instrSession, VI_ERROR_FAIL_ID_QUERY);
    }
        
    /*- Reset instrument ----------------------------------------------------*/
    if (resetDevice)
    {
        if ((SNSGA_status = SNSGA_reset (*instrSession)) != VI_SUCCESS)
            return SNSGA_initCleanUp (rmSession, instrSession, SNSGA_status);
    }       
          
    return SNSGA_status;
}

/*===========================================================================*/
/* Function: GenerateDCVolt                                                  */
/* Purpose:  This function sets the output voltage to the specified value.   */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_GenerateDCVolt (ViSession      instrSession,
                                       ViReal64       voltage)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK

    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE:VOLTAGE ");
    sprintf(tmpBuffer,"%f",voltage);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
}

/*===========================================================================*/
/* Function: GenerateDCCurr                                                  */
/* Purpose:  This function sets the output current to the specified value.   */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_GenerateDCCurr (ViSession      instrSession,
                                       ViReal64       current)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE:CURRENT ");
    sprintf(tmpBuffer,"%f",current);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
}

/*===========================================================================*/
/* Function: GenerateDCVRamp                                                 */
/* Purpose:  This function initates a output voltage ramp from the present   */
/*           output voltage to the specified output voltage in the specified */
/*           time on the specified channel. Time is in seconds.              */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_GenerateDCVRamp  (ViSession      instrSession,
                                         ViReal64       voltage,
                                         ViReal64       time_sec)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE:VOLTAGE:RAMP ");
    sprintf(tmpBuffer,"%f ",voltage);
    strcat(cmdBuffer,tmpBuffer);
    sprintf(tmpBuffer,"%f",time_sec);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
}
                                          
/*===========================================================================*/
/* Function: GenerateDCCRamp                                                 */
/* Purpose:  This function initates a output Current ramp from the present   */
/*           output voltage to the specified output current in the specified */
/*           time on the specified channel. Time is in seconds.              */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_GenerateDCCRamp  (ViSession      instrSession,
                                         ViReal64       current,
                                         ViReal64       time_sec)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE:CURRENT:RAMP ");
    sprintf(tmpBuffer,"%f ",current);
    strcat(cmdBuffer,tmpBuffer);
    sprintf(tmpBuffer,"%f",time_sec);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
}

/*===========================================================================*/
/* Function: OutState                                                       */
/* Purpose:  This function reads the output voltage.                         */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_OutState (ViSession      instrSession,
                                 ViBoolean      state)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"OUTPUT:STATE ");
    sprintf(tmpBuffer,"%d",state);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
    
}

/*===========================================================================*/
/* Function: MeasureDCVolt                                                   */
/* Purpose:  This function reads the output voltage.                         */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_MeasureDCVolt (ViSession      instrSession,
                                      ViReal64       *voltage)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"MEASURE:VOLTAGE?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
	if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *voltage = atof(rdBuffer);

	UNLOCK
}

/*===========================================================================*/
/* Function: MeasureDCCurr                                                   */
/* Purpose:  This function reads the output voltage.                         */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_MeasureDCCurr (ViSession      instrSession,
                                      ViReal64       *current)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"MEASURE:CURRENT?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *current = atof(rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: Mode                                                            */
/* Purpose:  This function returns the operating mode of the power supply.   */
/*           (voltage mode = 0, current mode = 1)                            */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ModeQuery (ViSession      instrSession,
                                  ViInt16        *mode)
{
    ViStatus		SNSGA_status,unlockStatus;
	unsigned char	protCondReg  = 0;
	ViByte			statusMessage[BUFFER_SIZE];

	LOCK
	
	if ((SNSGA_status = SNSGA_ProtCondQuery(instrSession, &protCondReg, statusMessage)) < 0)
	{
		*mode = -1;
		UNLOCK;
	}

	if (protCondReg & SNSGA_PE_CURRMODE)
		*mode = 1;
	else
		*mode = 0;
		
	UNLOCK
}

/*===========================================================================*/
/* Function: VoltMode                                                        */
/* Purpose:  This function configures the power supply to shutdown if the    */
/*           power supply switches from voltage mode to current mode.        */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_VoltMode (ViSession      instrSession,
                                 ViBoolean      serviceRequest)
                                
{
    ViStatus		SNSGA_status,unlockStatus;
	unsigned char	mask;

	LOCK
	
	mask =  SNSGA_PE_CURRMODE | SNSGA_PE_FOLD;
	
	/* program foldback */
	if ((SNSGA_status = SNSGA_ConfigFoldback (instrSession, SNSGA_FB_CURRMODE)) < 0)
		UNLOCK
		
	/* program protection event enable mask for foldback and current mode */
	if ((SNSGA_status = SNSGA_ConfigProtEventEnable (instrSession, mask)) < 0)
		UNLOCK
		
	if (serviceRequest)
	{
		/* program service request enable mask */
		SNSGA_status = SNSGA_ConfigSerReqEnable (instrSession, SNSGA_SB_SRQ);
	}
	
	UNLOCK
}

/*===========================================================================*/
/* Function: CurrMode                                                        */
/* Purpose:  This function configures the power supply to shutdown if the    */
/*           power supply switches from current mode to voltage mode.        */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_CurrMode (ViSession      instrSession,
                                 ViBoolean      serviceRequest)
                                
{
    ViStatus	SNSGA_status,unlockStatus;
	unsigned char	mask;

	LOCK
	
	mask =  SNSGA_PE_VOLTMODE | SNSGA_PE_FOLD;
	
	/* program foldback */
	if ((SNSGA_status = SNSGA_ConfigFoldback (instrSession, SNSGA_FB_VOLTMODE)) < 0)
		UNLOCK
		
	/* program protection event enable mask */
	if ((SNSGA_status = SNSGA_ConfigProtEventEnable (instrSession, mask)) < 0)
		UNLOCK
		
	if (serviceRequest)
	{
		/* program service request enable mask */
		SNSGA_status = SNSGA_ConfigSerReqEnable (instrSession, SNSGA_SB_SRQ);
	}
	
	UNLOCK    
}

/*===========================================================================*/
/* Function: NormalMode                                                      */
/* Purpose:  This function configures the power supply operate in both       */
/*           current mode and voltage mode. All protection events and        */
/*           service request enables are cleared.                            */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_NormalMode (ViSession      instrSession)
{
    ViStatus	SNSGA_status,unlockStatus;

	LOCK
	
	/* program foldback */
	if ((SNSGA_status = SNSGA_ConfigFoldback (instrSession, SNSGA_FB_NORMAL)) < 0)
		UNLOCK
		
	/* program protection event enable mask */
	if ((SNSGA_status = SNSGA_ConfigProtEventEnable (instrSession, 0)) < 0)
		UNLOCK
		
	/* program service request enable mask */
	SNSGA_status = SNSGA_ConfigSerReqEnable (instrSession, 0);

	UNLOCK    
}

/*===========================================================================*/
/* Function: ConfigDCVolt                                                    */
/* Purpose:  This function configures the power supply for a triggered       */
/*           output voltage of the specified value. A software trigggger     */
/*           must be sent to the power supply to implement this sequence.    */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ConfigDCVolt (ViSession      instrSession,
                                     ViReal64       voltage)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE:VOLTAGE:TRIGGERED ");
    sprintf(tmpBuffer,"%f",voltage);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}
                                          
/*===========================================================================*/
/* Function: ConfigDCCurr                                                    */
/* Purpose:  This function configures the power supply for a triggered       */
/*           output voltage of the specified value. A software trigggger     */
/*           must be sent to the power supply to implement this sequence.    */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ConfigDCCurr (ViSession      instrSession,
                                     ViReal64       current)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE:CURRENT:TRIGGERED ");
    sprintf(tmpBuffer,"%f",current);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigDCVRamp                                                   */
/* Purpose:  This function configures the power supply for a triggered       */
/*           output voltage ramp from the present output voltage to the      */
/*           specified voltage in the specified time on the specified        */
/*           channel. A software trigger must be sent to the power supply to */
/*           implement this sequence.                                        */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ConfigDCVRamp (ViSession      instrSession,
                                      ViReal64       voltage,
                                      ViReal64       time_sec)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE:VOLTAGE:RAMP:TRIGGERED ");
    sprintf(tmpBuffer,"%f ",voltage);
    strcat(cmdBuffer,tmpBuffer);
    sprintf(tmpBuffer,"%f",time_sec);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}
                                          
/*===========================================================================*/
/* Function: ConfigDCCRamp                                                   */
/* Purpose:  This function configures the power supply for a triggered       */
/*           output current ramp from the present output current to the      */
/*           specified current in the specified time on the specified        */
/*           channel. A software trigger must be sent to the power supply to */
/*           implement this sequence. Time is in seconds.                    */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ConfigDCCRamp (ViSession      instrSession,
                                      ViReal64       current,
                                      ViReal64       time_sec)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE:CURRENT:RAMP:TRIGGERED ");
    sprintf(tmpBuffer,"%f ",current);
    strcat(cmdBuffer,tmpBuffer);
    sprintf(tmpBuffer,"%f",time_sec);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigOVP                                                       */
/* Purpose:  This function sets the output overvoltage protection to the     */
/*           specified value.                                                */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ConfigOVP (ViSession      instrSession,
                                  ViReal64       voltage)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE:VOLTAGE:PROTECTION ");
    sprintf(tmpBuffer,"%f",voltage);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);

    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigSoftLimits                                                */
/* Purpose:  This function sets the soft voltage and current programming     */
/*           limits to the specified values.                                 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ConfigSoftLimits (ViSession      instrSession,
                                         ViReal64       voltage,
                                         ViReal64       current)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
	LOCK
	
    /* build and send SCPI command string - voltage */
    strcpy(cmdBuffer,"SOURCE:VOLTAGE:LIMIT ");
    sprintf(tmpBuffer,"%f",voltage);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* build and send SCPI command string - current */
    strcpy(cmdBuffer,"SOURCE:CURRENT:LIMIT ");
    sprintf(tmpBuffer,"%f",current);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigPowerON                                                   */
/* Purpose:  This function configures the output power on values for the     */
/*           specified channel.                                              */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ConfigPowerOn (ViSession		instrSession,
                                      ViReal64		voltage,
                                      ViReal64		current,
                                      ViReal64		ovp)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK
	
    /* build and send SCPI command string - voltage */
    strcpy(cmdBuffer,"CALIBRATE:INITIAL:VOLTAGE ");
    sprintf(tmpBuffer,"%f",voltage);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);

    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* build and send SCPI command string - current */
    strcpy(cmdBuffer,"CALIBRATE:INITIAL:CURRENT ");
    sprintf(tmpBuffer,"%f",current);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);

    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* build and send SCPI command string - ovp */
    strcpy(cmdBuffer,"CALIBRATE:INITIAL:VOLTAGE:PROTECTION ");
    sprintf(tmpBuffer,"%f",ovp);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);

    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* build and send SCPI command string - unlock */
    strcpy(cmdBuffer,"CALIBRATE:UNLOCK \"6867\"");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* build  and send SCPI command string - store */
    strcpy(cmdBuffer,"CALIBRATE:STORE");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* build and send SCPI command string - lock */
    strcpy(cmdBuffer,"CALIBRATE:LOCK");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigFoldback                                                  */
/* Purpose:  This function configures the foldback mode if any. Three modes  */
/*           are normal, constant current, and constant voltage.             */
/*           See header file for bit definitions.                            */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ConfigFoldback (ViSession      instrSession,
                                       ViInt16        mode)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"OUTPUT:PROTECTION:FOLD ");
    sprintf(tmpBuffer,"%d",mode);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigProtDelay                                                 */
/* Purpose:  This function configures the programmable delay which is        */
/*           executed prior to reporting any output protection conditions.   */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ConfigProtDelay (ViSession instrSession,
                                        ViReal64  delay)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"OUTPUT:PROTECTION:DELAY ");
    sprintf(tmpBuffer,"%f",delay);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigVoltAve                                                   */
/* Purpose:  This function configures the number of current measurents to be */
/*           averaged. 1-5                                                   */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ConfigVoltAve (ViSession instrSession,
                                      ViInt16   average)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"MEASURE:VOLTAGE:AVERAGE ");
    sprintf(tmpBuffer,"%d",average);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigCurrAve                                                   */
/* Purpose:  This function configures the number of current measurents to be */
/*           averaged. 1-5                                                   */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ConfigCurrAve (ViSession instrSession,
                                      ViInt16   average)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"MEASURE:CURRENT:AVERAGE ");
    sprintf(tmpBuffer,"%d",average);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigProtEventEnable                                           */
/* Purpose:  This function configures the protection event status enable     */
/*           register. See header file for bit definitions.                  */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ConfigProtEventEnable (ViSession      instrSession,
                                              unsigned char  protectionEventEnable)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"STATUS:PROTECTION:ENABLE ");
    sprintf(tmpBuffer,"%d",protectionEventEnable);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigStandEventEnable                                          */
/* Purpose:  This function configures the standard event status enable       */
/*           register. See header file for bit definitions.                  */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ConfigStandEventEnable (ViSession      instrSession,
                                               unsigned char  standardEventEnable)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"*ESE");
    sprintf(tmpBuffer," %d",standardEventEnable);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigSerReqEnable                                              */
/* Purpose:  This function configures the service request enable             */
/*           register. See header file for bit definitions.                  */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ConfigSerReqEnable (ViSession      instrSession,
                                           unsigned char  serviceRequestEnable)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"*SRE");
    sprintf(tmpBuffer," %d",serviceRequestEnable);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: OutputVoltQuery                                                */
/* Purpose:  This function returns the programmed output voltage value.      */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_OutputVoltQuery  (ViSession      instrSession,
                                         ViReal64       *voltage)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE:VOLTAGE?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query string */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *voltage = atof(rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: OutputCurrQuery                                                */
/* Purpose:  This function returns the programmed output voltage value.      */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_OutputCurrQuery  (ViSession      instrSession,
                                         ViReal64       *current)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE:CURRENT?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

	/* get returnd query string */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *current = atof(rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: OutputOVPQuery                                                  */
/* Purpose:  This function returns the programmed output voltage value.      */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_OutputOVPQuery (ViSession      instrSession,
                                       ViReal64       *OVP)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE:VOLTAGE:PROTECTION?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get return query string */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *OVP = atof(rdBuffer);

    UNLOCK
}
                                        

/*===========================================================================*/
/* Function: OutStateQuery                                                   */
/* Purpose:  This function returns the output state, 1 = ON.                 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_OutStateQuery (ViSession      instrSession,
                                      ViBoolean      *state)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string - voltage limit */
    strcpy(cmdBuffer,"OUTPUT:STATE?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query string */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *state = atoi(rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: SoftLimitsQuery                                                 */
/* Purpose:  This function returns the voltage and current soft limits.      */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_SoftLimitsQuery  (ViSession      instrSession,
                                         ViReal64       *voltage,
                                         ViReal64       *current)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string - voltage limit */
    strcpy(cmdBuffer,"SOURCE:VOLTAGE:LIMIT?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query string */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *voltage = atof(rdBuffer);

	/* build and send SCPI command string - current limit */
    strcpy(cmdBuffer,"SOURCE:CURRENT:LIMIT?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query string */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *current = atof(rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: PowerONQuery                                                    */
/* Purpose:  This function returns the output power on values for the        */
/*           specified channel.                                              */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_PowerOnQuery  (ViSession      instrSession,
                                      ViReal64       *voltage,
                                      ViReal64       *current,
                                      ViReal64       *ovp)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];
    
	LOCK
	
	/* build and send SCPI command string - power on voltage */
    strcpy(cmdBuffer,"CALIBRATE:INITIAL:VOLTAGE?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query string */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *voltage = atof(rdBuffer);

	/* build and send SCPI comand string - power on current */
    strcpy(cmdBuffer,"CALIBRATE:INITIAL:CURRENT?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query string */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *current = atof(rdBuffer);

	/* build and send SCPI command string - power on ovp */
    strcpy(cmdBuffer,"CALIBRATE:INITIAL:VOLTAGE:PROTECTION?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query string */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *ovp = atof(rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: protDelayQuery                                                  */
/* Purpose:  This function returns the protection condition delay value      */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_protDelayQuery (ViSession instrSession,
                                       ViReal64  *delay)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string - voltage limit */
    strcpy(cmdBuffer,"OUTPUT:PROTECTION:DELAY?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query string */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *delay = atof(rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: measVoltAveQuery                                                */
/* Purpose:  This function returns the number of voltage readings that will  */
/*           be averaged upon receipt of a voltage measurent command for the */
/*           specified channel.                                              */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_measVoltAveQuery (ViSession instrSession,
                                         ViInt16   *average)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string - voltage limit */
    strcpy(cmdBuffer,"MEASURE:VOLTAGE:AVERAGE?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query string */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *average = atoi(rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: measCurrAveQuery                                                */
/* Purpose:  This function returns the number of current readings that will  */
/*           be averaged upon receipt of a current measurent command for the */
/*           specified channel.                                              */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_measCurrAveQuery (ViSession instrSession,
                                         ViInt16   *average)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string - voltage limit */
    strcpy(cmdBuffer,"MEASURE:CURRENT:AVERAGE?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query string */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *average = atoi(rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: SendSWTrigger                                                   */
/* Purpose:  This function sends a software trigger to the specified         */
/*           channel. Setting the channel to zero causes a Group Execute     */
/*           Trigger for all power supplies that have been previously        */
/*           configured for triggered outputs.                               */
/*           Trigger Types: 1 = Voltage, 2 = Current, 3 = Both, 4 = Ramp     */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_SendSWTrigger (ViSession      instrSession,
                                      ViInt16        type)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK
	
	/* test for voltage, current, or both trigger type */
    if ((type > 0) && (type < 4))
    {
		/* build SCPI command string */
	    strcpy(cmdBuffer,"TRIGGER:TYPE ");
	    sprintf(tmpBuffer,"%d",type);
	    strcat(cmdBuffer,tmpBuffer);
	}
	/* test for ramp trigger type */
	else if (type == 4)
    {
		/* build SCPI command string */
	    strcpy(cmdBuffer,"TRIGGER:RAMP");
	}
	else
		return VI_ERROR_INV_PARAMETER;
	
	/* send SCPI command string */
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    UNLOCK
}

/*===========================================================================*/
/* Function: Abort Trigger                                                   */
/* Purpose:  This function clears all previously configured voltage and      */
/*           current triggers for the specified channel.                     */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_AbortTrigger (ViSession      instrSession)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];

	LOCK
	
	/* build and send SCPI command string */
    strcpy(cmdBuffer,"TRIGGER:ABORT");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: StatusByteQuery                                                 */
/* Purpose:  This function returns the instruments status byte and a message */
/*           indicating which bits are active.                               */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_StatusByteQuery (ViSession      instrSession,
                                        unsigned char  *statusByte,
                                        ViChar _VI_FAR statusMessage[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViChar		rdBuffer[BUFFER_SIZE];
    char		*endPtr;

	LOCK
	
	/* build and send SCPI command string */
    strcpy(cmdBuffer,"*STB?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query string */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* assign value */
	*statusByte = (unsigned char) strtoul((const char *) rdBuffer, &endPtr, 10);

	/* generate status message from status byte bits, bits 0,3,7 are NOT USED */
	if (*statusByte == 0)
		strcpy(statusMessage,"00");
	else
	{
		sprintf(rdBuffer,"%02X - ",*statusByte);
		if (*statusByte & SNSGA_SB_PROTEVENT)
			strcat(rdBuffer,"Protection Event, ");
		if (*statusByte & SNSGA_SB_ERRORQUERY)
			strcat(rdBuffer,"Error Event Message, ");
		if (*statusByte & SNSGA_SB_MAV)
			strcat(rdBuffer,"Message Available, ");
		if (*statusByte & SNSGA_SB_STANDEVENT)
			strcat(rdBuffer,"Event Status, ");
		if (*statusByte & SNSGA_SB_SRQ)
			strcat(rdBuffer,"Request Service");
		
		/* remove any ending ',' */
		retCnt = strlen(rdBuffer);
		if (rdBuffer[retCnt - 2] == ',')
			rdBuffer[retCnt - 2] = '\0';

		strcpy(statusMessage,rdBuffer);
	}
	
    UNLOCK
}

/*===========================================================================*/
/* Function: StandEventQuery                                                 */
/* Purpose:  This function returns the instruments standard event status     */
/*           register and a message indicating which bits are active.        */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_StandEventQuery (ViSession      instrSession,
                                        unsigned char  *standardEventStatusRegister,
                                        ViChar _VI_FAR statusMessage[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViChar		rdBuffer[BUFFER_SIZE];
    char		*endPtr;

	LOCK
	
	/* build and send SCPI command string */
    strcpy(cmdBuffer,"*ESR?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query string */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* assign value */
	*standardEventStatusRegister = (unsigned char) strtoul((const char *) rdBuffer, &endPtr, 10);

	/* generate status message from register bits, bits 1,6 are NOT USED */
	if (*standardEventStatusRegister == 0)
		strcpy(statusMessage,"00");
	else
	{
		sprintf(rdBuffer,"%02X - ",*standardEventStatusRegister);
		if (*standardEventStatusRegister & SNSGA_SE_OPC)
			strcat(rdBuffer,"Operation Complete, ");
		if (*standardEventStatusRegister & SNSGA_SE_ERRORQUERY)
			strcat(rdBuffer,"Query Error, ");
		if (*standardEventStatusRegister & SNSGA_SE_DEVICEERROR)
			strcat(rdBuffer,"Device Error, ");
		if (*standardEventStatusRegister & SNSGA_SE_RANGEERROR)
			strcat(rdBuffer,"Execution Error (Range), ");
		if (*standardEventStatusRegister & SNSGA_SE_SYNTAXERROR)
			strcat(rdBuffer,"Command Error (Syntax), ");
		if (*standardEventStatusRegister & SNSGA_SE_PWRON)
			strcat(rdBuffer,"Power On");
		
		/* remove any ending ',' */
		retCnt = strlen(rdBuffer);
		if (rdBuffer[retCnt - 2] == ',')
			rdBuffer[retCnt - 2] = '\0';

		strcpy(statusMessage,rdBuffer);
	}
	
    UNLOCK
}

/*===========================================================================*/
/* Function: StandEventEnableQuery                                           */
/* Purpose:  This function returns the instruments standard event enable     */
/*           register and a message indicating which bits are active.        */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_StandEventEnableQuery (ViSession      instrSession,
                                              unsigned char  *standardEventEnableRegister,
                                              ViChar _VI_FAR statusMessage[])
{
    
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViChar		rdBuffer[BUFFER_SIZE];
    char		*endPtr;

	LOCK
	
	/* build and send SCPI command string */
    strcpy(cmdBuffer,"*ESE?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query string */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* assign value */
	*standardEventEnableRegister = (unsigned char) strtoul((const char *) rdBuffer, &endPtr, 10);

	/* generate status message from register bits, bits 1,6 are NOT USED */
	if (*standardEventEnableRegister == 0)
		strcpy(statusMessage,"00");
	else
	{
		sprintf(rdBuffer,"%02X - ",*standardEventEnableRegister);
		if (*standardEventEnableRegister & SNSGA_SE_OPC)
			strcat(rdBuffer,"Operation Complete, ");
		if (*standardEventEnableRegister & SNSGA_SE_ERRORQUERY)
			strcat(rdBuffer,"Query Error, ");
		if (*standardEventEnableRegister & SNSGA_SE_DEVICEERROR)
			strcat(rdBuffer,"Device Error, ");
		if (*standardEventEnableRegister & SNSGA_SE_RANGEERROR)
			strcat(rdBuffer,"Execution Error (Range), ");
		if (*standardEventEnableRegister & SNSGA_SE_SYNTAXERROR)
			strcat(rdBuffer,"Command Error (Syntax), ");
		if (*standardEventEnableRegister & SNSGA_SE_PWRON)
			strcat(rdBuffer,"Power On");
		
		/* remove any ending ',' */
		retCnt = strlen(rdBuffer);
		if (rdBuffer[retCnt - 2] == ',')
			rdBuffer[retCnt - 2] = '\0';

		strcpy(statusMessage,rdBuffer);
	}
	
    UNLOCK
}

/*===========================================================================*/
/* Function: ProtEventQuery                                                  */
/* Purpose:  This function returns the instruments protection event status   */
/*           register and a message indicating which bits are active.        */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ProtEventQuery (ViSession      instrSession,
                                       unsigned char  *protEventReg,
                                       ViChar _VI_FAR statusMessage[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViChar		rdBuffer[BUFFER_SIZE];
    char		*endPtr;

	LOCK
	
	/* build and send SCPI command string */
    strcpy(cmdBuffer,"STATUS:PROTECTION:EVENT?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query string */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* assign value */
	*protEventReg = (unsigned char) strtoul((const char *) rdBuffer, &endPtr, 10);

	/* generate status message from register bits */
	if (*protEventReg == 0)
		strcpy(statusMessage,"00");
	else
	{
		sprintf(rdBuffer,"%02X - ",*protEventReg);
		if (*protEventReg & SNSGA_PE_VOLTMODE)
			strcat(rdBuffer,"Constant Voltage, ");
		if (*protEventReg & SNSGA_PE_CURRMODE)
			strcat(rdBuffer,"Constant Current, ");
		if (*protEventReg & SNSGA_PE_HW)
			strcat(rdBuffer,"Hardware Fault, ");
		if (*protEventReg & SNSGA_PE_OVP)
			strcat(rdBuffer,"OVP, ");
		if (*protEventReg & SNSGA_PE_OTP)
			strcat(rdBuffer,"OTP, ");
		if (*protEventReg & SNSGA_PE_EXTSD)
			strcat(rdBuffer,"Ext. Shutdown, ");
		if (*protEventReg & SNSGA_PE_FOLD)
			strcat(rdBuffer,"Foldback Mode, ");
		if (*protEventReg & SNSGA_PE_PRGMERROR)
			strcat(rdBuffer,"Programming Error");
		
		/* remove any ending ',' */
		retCnt = strlen(rdBuffer);
		if (rdBuffer[retCnt - 2] == ',')
			rdBuffer[retCnt - 2] = '\0';

		strcpy(statusMessage,rdBuffer);
	}
	
    UNLOCK
}

/*===========================================================================*/
/* Function: ProtEventEnableQuery                                            */
/* Purpose:  This function returns the instruments protection event enable   */
/*           register and a message indicating which bits are active.        */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ProtEventEnableQuery (ViSession      instrSession,
                                             unsigned char  *protEventEnableRegister,
                                             ViChar _VI_FAR statusMessage[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViChar		rdBuffer[BUFFER_SIZE];
    char		*endPtr;

	LOCK
	
	/* build and send SCPI command string */
    strcpy(cmdBuffer,"STATUS:PROTECTION:ENABLE?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query string */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* assign value */
	*protEventEnableRegister = (unsigned char) strtoul((const char *) rdBuffer, &endPtr, 10);

	/* generate status message from register bits */
	if (*protEventEnableRegister == 0)
		strcpy(statusMessage,"00");
	else
	{
		sprintf(rdBuffer,"%02X - ",*protEventEnableRegister);
		if (*protEventEnableRegister & SNSGA_PE_VOLTMODE)
			strcat(rdBuffer,"Constant Voltage, ");
		if (*protEventEnableRegister & SNSGA_PE_CURRMODE)
			strcat(rdBuffer,"Constant Current, ");
		if (*protEventEnableRegister & SNSGA_PE_HW)
			strcat(rdBuffer,"Hardware Fault, ");
		if (*protEventEnableRegister & SNSGA_PE_OVP)
			strcat(rdBuffer,"OVP, ");
		if (*protEventEnableRegister & SNSGA_PE_OTP)
			strcat(rdBuffer,"OTP, ");
		if (*protEventEnableRegister & SNSGA_PE_EXTSD)
			strcat(rdBuffer,"Ext. Shutdown, ");
		if (*protEventEnableRegister & SNSGA_PE_FOLD)
			strcat(rdBuffer,"Foldback Mode, ");
		if (*protEventEnableRegister & SNSGA_PE_PRGMERROR)
			strcat(rdBuffer,"Programming Error");
		
		/* remove any ending ',' */
		retCnt = strlen(rdBuffer);
		if (rdBuffer[retCnt - 2] == ',')
			rdBuffer[retCnt - 2] = '\0';

		strcpy(statusMessage,rdBuffer);
	}
	
    UNLOCK
}

/*===========================================================================*/
/* Function: ProtCondQuery                                                   */
/* Purpose:  This function returns the instruments protection condition      */
/*           status register and a message indicating which bits are active. */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ProtCondQuery (ViSession      instrSession,
                                      unsigned char  *protCondReg,
                                      ViChar _VI_FAR statusMessage[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViChar		rdBuffer[BUFFER_SIZE];
    char		*endPtr;

	LOCK
	
	/* build and send SCPI command string */
    strcpy(cmdBuffer,"STATUS:PROTECTION:CONDITION?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query string */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* assign value */
	*protCondReg = (unsigned char) strtoul((const char *) rdBuffer, &endPtr, 10);

	/* generate status message from register bits */
	if (*protCondReg == 0)
		strcpy(statusMessage,"00");
	else
	{
		sprintf(rdBuffer,"%02X - ",*protCondReg);
		if (*protCondReg & SNSGA_PE_VOLTMODE)
			strcat(rdBuffer,"Constant Voltage, ");
		if (*protCondReg & SNSGA_PE_CURRMODE)
			strcat(rdBuffer,"Constant Current, ");
		if (*protCondReg & SNSGA_PE_HW)
			strcat(rdBuffer,"Hardware Fault, ");
		if (*protCondReg & SNSGA_PE_OVP)
			strcat(rdBuffer,"OVP, ");
		if (*protCondReg & SNSGA_PE_OTP)
			strcat(rdBuffer,"OTP, ");
		if (*protCondReg & SNSGA_PE_EXTSD)
			strcat(rdBuffer,"Ext. Shutdown, ");
		if (*protCondReg & SNSGA_PE_FOLD)
			strcat(rdBuffer,"Foldback Mode, ");
		if (*protCondReg & SNSGA_PE_PRGMERROR)
			strcat(rdBuffer,"Programming Error");
		
		/* remove any ending ',' */
		retCnt = strlen(rdBuffer);
		if (rdBuffer[retCnt - 2] == ',')
			rdBuffer[retCnt - 2] = '\0';

		strcpy(statusMessage,rdBuffer);
	}
	
    UNLOCK
}

/*===========================================================================*/
/* Function: ServiceReqEnableQuery                                           */
/* Purpose:  This function returns the instruments service request enable    */
/*           register and a message indicating which bits are active.        */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ServiceReqEnableQuery (ViSession      instrSession,
                                              unsigned char  *serviceReqEnableRegister,
                                              ViChar _VI_FAR statusMessage[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViChar		rdBuffer[BUFFER_SIZE];
    char		*endPtr;

	LOCK
	
	/* build and send SCPI command string */
    strcpy(cmdBuffer,"*SRE?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query string */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* assign value */
	*serviceReqEnableRegister = (unsigned char) strtoul((const char *) rdBuffer, &endPtr, 10);

	/* generate status message from status byte bits, bits 0,3,7 are NOT USED */
	if (*serviceReqEnableRegister == 0)
		strcpy(statusMessage,"00");
	else
	{
		sprintf(rdBuffer,"%02X - ",*serviceReqEnableRegister);
		if (*serviceReqEnableRegister & 0x02)
			strcat(rdBuffer,"Protection Event, ");
		if (*serviceReqEnableRegister & 0x04)
			strcat(rdBuffer,"Error Event Message, ");
		if (*serviceReqEnableRegister & 0x10)
			strcat(rdBuffer,"Message Available, ");
		if (*serviceReqEnableRegister & 0x20)
			strcat(rdBuffer,"Event Status, ");
		if (*serviceReqEnableRegister & 0x40)
			strcat(rdBuffer,"Request Service");
		
		/* remove any ending ',' */
		retCnt = strlen(rdBuffer);
		if (rdBuffer[retCnt - 2] == ',')
			rdBuffer[retCnt - 2] = '\0';

		strcpy(statusMessage,rdBuffer);
	}
	
    UNLOCK
}

/*===========================================================================*/
/* Function: Force PS to configure "\r" for querying data                                                           */
/* Purpose:  This function configures the instrument to a produce a "\r"     */
/*===========================================================================*/
/*ViStatus CVIFUNC SNSGA_configTerm (ViSession instrSession,
                                      ViInt16   termChar)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK
	
    // build and send SCPI command string 
    strcpy(cmdBuffer,"SYST:NET:TERM ");
    sprintf(tmpBuffer,"%d",termChar);
    strcat(cmdBuffer,tmpBuffer);
	
    // add <CR> to end of command for ENET_IF 
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);
	
	if( termChar == 1)
	{
		viSetAttribute (instrSession, VI_ATTR_TERMCHAR, '\r');  	
	}
	else if( termChar == 2)   
	{
		viSetAttribute (instrSession, VI_ATTR_TERMCHAR, '\n');  	
	}
	
    UNLOCK
} */
	
/*===========================================================================*/
/* Function: Reset                                                           */
/* Purpose:  This function resets the instrument to a known state.           */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_reset (ViSession      instrSession)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];

	LOCK
	
    /*  send SCPI reset command */
    strcpy(cmdBuffer,"*RST");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    UNLOCK
}

/*===========================================================================*/
/* Function: Clear                                                          */
/* Purpose:  This function clear the instrument error and status info.       */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_clear (ViSession      instrSession)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];

	LOCK
	
	/* send SCPI clear command */
    strcpy(cmdBuffer,"*CLS");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: Error Query                                                     */
/* Purpose:  This function queries the instrument error queue, and returns   */
/*           the result.                                                     */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_errorQuery (ViSession      instrSession,
                                   ViPInt32       errorCode,
                                   ViChar _VI_FAR errorMessage[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViUInt32	index;
    ViUInt32	i;
    ViStatus	SNSGA_status,unlockStatus;
    ViChar		*cptr;
    ViChar		rdBuffer[BUFFER_SIZE];
    ViByte		cmdBuffer[80];

	LOCK
	
    /*  send SCPI reset command */
    strcpy(cmdBuffer,"SYSTEM:ERROR?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

	/* get returned query string */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* get error code from string */
	sscanf(rdBuffer,"%d",errorCode);
	
	/* get error message, skip ", " in return string */
	cptr  = strchr(rdBuffer,',');
	cptr += 2;
	strcpy(errorMessage,cptr);

	UNLOCK
}

/*===========================================================================*/
/* Function: Error Message                                                   */
/* Purpose:  This function translates the error return value from the        */
/*           instrument driver into a user-readable string.                  */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_errorMessage (ViSession      instrSession,
                                     ViStatus       statusCode,
                                     ViChar _VI_FAR message[])
{
    ViStatus SNSGA_status,unlockStatus;
    ViInt16 i;
    static SNSGA_tStringValPair statusDescArray[] = {
        {VI_WARN_NSUP_ID_QUERY,                "WARNING: ID Query not supported"},
        {VI_WARN_NSUP_RESET,                   "WARNING: Reset not supported"},
        {VI_WARN_NSUP_SELF_TEST,               "WARNING: Self-test not supported"},
        {VI_WARN_NSUP_ERROR_QUERY,             "WARNING: Error Query not supported"},     
        {VI_WARN_NSUP_REV_QUERY,               "WARNING: Revision Query not supported"},
        {VI_ERROR_PARAMETER1,                  "ERROR: Parameter 1 out of range"},
        {VI_ERROR_PARAMETER2,                  "ERROR: Parameter 2 out of range"},
        {VI_ERROR_PARAMETER3,                  "ERROR: Parameter 3 out of range"},
        {VI_ERROR_PARAMETER4,                  "ERROR: Parameter 4 out of range"},
        {VI_ERROR_PARAMETER5,                  "ERROR: Parameter 5 out of range"},
        {VI_ERROR_PARAMETER6,                  "ERROR: Parameter 6 out of range"},
        {VI_ERROR_PARAMETER7,                  "ERROR: Parameter 7 out of range"},
        {VI_ERROR_PARAMETER8,                  "ERROR: Parameter 8 out of range"},
        {VI_ERROR_FAIL_ID_QUERY,               "ERROR: Identification query failed"},
        {VI_ERROR_INV_RESPONSE,                "ERROR: Interpreting instrument response"},
        {VI_ERROR_INSTR_FILE_OPEN,             "ERROR: Opening the specified file"},
        {VI_ERROR_INSTR_FILE_WRITE,            "ERROR: Writing to the specified file"},
        {VI_ERROR_INSTR_INTERPRETING_RESPONSE, "ERROR: Interpreting the instrument's response"},
        
        /* not used, example of user defined Status Code
        {PREFIX_ERROR_INVALID_CONFIGURATION,   "ERROR: Instrument configuration error"},
        */
        
        {VI_NULL, VI_NULL}
    };

    SNSGA_status = viStatusDesc (instrSession, statusCode, message);
    if (SNSGA_status == VI_WARN_UNKNOWN_STATUS) {
        for (i=0; statusDescArray[i].stringName; i++) {
            if (statusDescArray[i].stringVal == statusCode) {
                strcpy (message, statusDescArray[i].stringName);
                return (VI_SUCCESS);
            }
        }
        sprintf (message, "Unknown Error 0x%08lX", statusCode);
        return (VI_WARN_UNKNOWN_STATUS);
    }
    
    UNLOCK
}

/*===========================================================================*/
/* Function: revisionQuery                                                   */
/* Purpose:  This function returns the driver and instrument revisions.      */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_revisionQuery (ViSession      instrSession,
                                      ViChar _VI_FAR driverRev[],
                                      ViChar _VI_FAR instrRev[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViInt16		i;
    ViChar		*cptr;
    ViByte		rdBuffer[BUFFER_SIZE];
    ViByte		cmdBuffer[BUFFER_SIZE];

	LOCK
	
	/* send SCPI identify command */
	strcpy(cmdBuffer,"*IDN?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
	sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

    /* parse *idn? return string for firware revisions - 4 & 5 fields */
	cptr = rdBuffer;
	for (i = 0; i < 3; ++i)
	{
		cptr = strchr(cptr,',');
		++cptr;
	}
	
	/* copy driver and intrument strings to buffers */
	strcpy(instrRev, cptr);
    strcpy (driverRev, SNSGA_REVISION);

	UNLOCK
}

/*===========================================================================*/
/* Function: Identify Query                                                  */
/* Purpose:  This function returns the *idn? response.                       */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_idQuery (ViSession      instrSession,
                                ViChar _VI_FAR idBuffer[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViInt16		i;
    ViChar		*cptr;
    ViByte		rdBuffer[BUFFER_SIZE];
    ViByte		cmdBuffer[80];

	LOCK
	
	/* send SCPI identify command */
	strcpy(cmdBuffer,"*IDN?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
	sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* copy id string to buffer */
	strcpy(idBuffer, rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: ModelQuery                                                   */
/* Purpose:  This function reads the instrument's Maximum Voltage and        */
/*           Current for the specified channel.                              */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ModelQuery    (ViSession      instrSession,
                                      ViReal64       *modelVolt,
                                      ViReal64       *modelCurr)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    *modelVolt = 0.0;
    *modelCurr = 0.0;

    /* build and send SCPI command string - voltage */
    strcpy(cmdBuffer,"CALIBRATE:MODEL:VOLTAGE?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *modelVolt = atof(rdBuffer);

    /* build and send SCPI command string - current */
    strcpy(cmdBuffer,"CALIBRATE:MODEL:CURRENT?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *modelCurr = atof(rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: Close                                                           */
/* Purpose:  This function closes the instrument.                            */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_close (ViSession instrSession)
{
    SNSGA_instrRange instrPtr;
    ViSession rmSession;
    ViStatus SNSGA_status;

    if ((SNSGA_status = viGetAttribute (instrSession, VI_ATTR_RM_SESSION, &rmSession)) < 0)
		return SNSGA_status;
 	if ((SNSGA_status = viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr)) < 0)
		return SNSGA_status;
            
    if (instrPtr != NULL) 
        free (instrPtr);
    
    SNSGA_status = viClose (instrSession);
    viClose (rmSession);

	return SNSGA_status;
}

/*****************************************************************************/
/*= UTILITY ROUTINES (Non-Exportable Functions) =============================*/
/*****************************************************************************/

/*===========================================================================*/
/* Function: Boolean Value Out Of Range - ViBoolean                          */
/* Purpose:  This function checks a Boolean to see if it is equal to VI_TRUE */
/*           or VI_FALSE. If the value is out of range, the return value is  */
/*           VI_TRUE, otherwise the return value is VI_FALSE.                */
/*===========================================================================*/
ViBoolean SNSGA_invalidViBooleanRange (ViBoolean val)
{
    return ((val != VI_FALSE && val != VI_TRUE) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Short Signed Integer Value Out Of Range - ViInt16               */
/* Purpose:  This function checks a short signed integer value to see if it  */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range, the return value is VI_TRUE, otherwise the return     */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean SNSGA_invalidViInt16Range (ViInt16 val, ViInt16 min, ViInt16 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Long Signed Integer Value Out Of Range - ViInt32                */
/* Purpose:  This function checks a long signed integer value to see if it   */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range, the return value is VI_TRUE, otherwise the return     */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean SNSGA_invalidViInt32Range (ViInt32 val, ViInt32 min, ViInt32 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Unsigned Char Value Out Of Range - ViUInt8                      */
/* Purpose:  This function checks an unsigned char value to see if it        */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range, the return value is VI_TRUE, otherwise the return     */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean SNSGA_invalidViUInt8Range (ViUInt8 val, ViUInt8 min, ViUInt8 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Short Unsigned Integer Value Out Of Range - ViUInt16            */
/* Purpose:  This function checks a short unsigned integer value to see if it*/  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range, the return value is VI_TRUE, otherwise the return     */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean SNSGA_invalidViUInt16Range (ViUInt16 val, ViUInt16 min, ViUInt16 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Long Unsigned Integer Value Out Of Range - ViUInt32             */
/* Purpose:  This function checks a long unsigned integer value to see if it */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range, the return value is VI_TRUE, otherwise the return     */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean SNSGA_invalidViUInt32Range (ViUInt32 val, ViUInt32 min, ViUInt32 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Real (Float) Value Out Of Range - ViReal32                      */
/* Purpose:  This function checks a real (float) value to see if it lies     */  
/*           between a minimum and maximum value.  If the value is out of    */
/*           range, the return value is VI_TRUE, otherwise the return value  */
/*           is VI_FALSE.                                                    */
/*===========================================================================*/
ViBoolean SNSGA_invalidViReal32Range (ViReal32 val, ViReal32 min, ViReal32 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Real (Double) Value Out Of Range - ViReal64                     */
/* Purpose:  This function checks a real (double) value to see if it lies    */  
/*           between a minimum and maximum value.  If the value is out of    */
/*           range, the return value is VI_TRUE, otherwise the return value  */
/*           is VI_FALSE.                                                    */
/*===========================================================================*/
ViBoolean SNSGA_invalidViReal64Range (ViReal64 val, ViReal64 min, ViReal64 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Initialize Clean Up                                             */
/* Purpose:  This function is used only by the SNSGA_init function.  When   */
/*           an error is detected this function is called to close the       */
/*           open resource manager and instrument object sessions and to     */
/*           set the instrSession that is returned from SNSGA_init to       */
/*           VI_NULL.                                                        */
/*===========================================================================*/
ViStatus SNSGA_initCleanUp (ViSession  openRMSession,
                            ViPSession openInstrSession,
                            ViStatus   currentStatus)
{
    viClose (*openInstrSession);
    viClose (openRMSession);
    *openInstrSession = VI_NULL;
    
    return currentStatus;
}

// TCP/IP commands 8-14-06 // RBF:


/*===========================================================================*/
/* Function: ConfigIp                                                        */
/* Purpose:  This function sets the IP address to the specified value.       */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ConfigIp (ViSession       instrSession,
	                                ViChar _VI_FAR  channelName[],
	                                ViChar _VI_FAR  ipSetAddress[])
{
   	ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":NET:IP ");
    strcat(cmdBuffer,ipSetAddress);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: IpQuery                                                         */
/* Purpose:  This function reads the IP address.                             */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_IpQuery (ViSession      instrSession,
                                   ViChar _VI_FAR channelName[],
                                   ViChar _VI_FAR ipGetAddress[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViInt16		i;
    ViChar		*cptr;
    ViByte		rdBuffer[BUFFER_SIZE];
    ViByte		cmdBuffer[80];

	LOCK
	
	/* send SCPI identify command */
	strcpy(cmdBuffer,"SYST");
	strcat(cmdBuffer,channelName);
	strcat(cmdBuffer,":NET:IP");
	strcat(cmdBuffer,"?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
	sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* copy ip string to buffer */
	strcpy(ipGetAddress, rdBuffer);

    UNLOCK
}
                                                  
/*===========================================================================*/
/* Function: ConfigMask                                                      */
/* Purpose:  This function sets the mask address to the specified value.     */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ConfigMask (ViSession      instrSession,
	                                  ViChar _VI_FAR channelName[],
	                                  ViChar _VI_FAR maskSetAddress[])
{
   	ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":NET:MASK ");
    strcat(cmdBuffer,maskSetAddress);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: IpQuery                                                         */
/* Purpose:  This function reads the Mask address.                           */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_MaskQuery (ViSession      instrSession,
                                     ViChar _VI_FAR channelName[],
                                     ViChar _VI_FAR maskGetAddress[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViInt16		i;
    ViChar		*cptr;
    ViByte		rdBuffer[BUFFER_SIZE];
    ViByte		cmdBuffer[80];

	LOCK
	
	/* send SCPI identify command */
	strcpy(cmdBuffer,"SYST");
	strcat(cmdBuffer,channelName);
	strcat(cmdBuffer,":NET:MASK");
	strcat(cmdBuffer,"?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
	sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* copy ip string to buffer */
	strcpy(maskGetAddress, rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigGate                                                      */
/* Purpose:  This function sets the Gate address to the specified value.     */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ConfigGate (ViSession       instrSession,
	                                  ViChar _VI_FAR  channelName[],
	                                  ViChar _VI_FAR  gateSetAddress[])
{
   	ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":NET:GATE ");
    strcat(cmdBuffer,gateSetAddress);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: GateQuery                                                       */
/* Purpose:  This function reads the Gate address    .                       */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_GateQuery (ViSession      instrSession,
                                     ViChar _VI_FAR channelName[],
                                     ViChar _VI_FAR gateGetAddress[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViInt16		i;
    ViChar		*cptr;
    ViByte		rdBuffer[BUFFER_SIZE];
    ViByte		cmdBuffer[80];

	LOCK
	
	/* send SCPI identify command */
	strcpy(cmdBuffer,"SYST");
	strcat(cmdBuffer,channelName);
	strcat(cmdBuffer,":NET:GATE");
	strcat(cmdBuffer,"?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
	sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* copy ip string to buffer */
	strcpy(gateGetAddress, rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigDNS                                                       */
/* Purpose:  This function sets the DNS address to the specified value.      */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ConfigDNS (ViSession       instrSession,
	                                 ViChar _VI_FAR  channelName[],
	                                 ViChar _VI_FAR  dnsSetAddress[])
{
   	ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":NET:DNS ");
    strcat(cmdBuffer,dnsSetAddress);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: DNSQuery                                                        */
/* Purpose:  This function reads the DNS address.                            */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_DNSQuery (ViSession      instrSession,
                                    ViChar _VI_FAR channelName[],
                                    ViChar _VI_FAR dnsGetAddress[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViInt16		i;
    ViChar		*cptr;
    ViByte		rdBuffer[BUFFER_SIZE];
    ViByte		cmdBuffer[80];

	LOCK
	
	/* send SCPI identify command */
	strcpy(cmdBuffer,"SYST");
	strcat(cmdBuffer,channelName);
	strcat(cmdBuffer,":NET:DNS");
	strcat(cmdBuffer,"?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
	sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* copy ip string to buffer */
	strcpy(dnsGetAddress, rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigPort                                                      */
/* Purpose:  This function sets the listen port to the specified value.      */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ConfigPort (ViSession      instrSession,
                                      ViChar _VI_FAR channelName[],
                                      ViInt32       listenPort)
{
   ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":NET:PORT ");
    sprintf(tmpBuffer,"%d",listenPort);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: IpQuery                                                         */
/* Purpose:  This function reads the listen port.                            */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_PortQuery (ViSession      instrSession,
                                     ViChar _VI_FAR channelName[],
                                     ViInt32        *listenPort)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string - listen port */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":NET:PORT?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query string */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *listenPort = atoi(rdBuffer);

    UNLOCK
}

                                                  
/*===========================================================================*/
/* Function: ConfigTerm                                                      */
/* Purpose:  This function sets the terminator attribute					 */
/*           to the specified value.                                         */
/*           1 = CR, 2 = LF, 3 = CR LF, 4 = LF CR  							 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ConfigTerm (ViSession      instrSession,
                                      ViChar _VI_FAR channelName[],
                                      ViInt32        terminator)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK

    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":NET:TERM ");
    sprintf(tmpBuffer,"%d",terminator);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	if( terminator == 1)
	{
		viSetAttribute (instrSession, VI_ATTR_TERMCHAR, '\r');  	
	}
	else if( terminator == 2)   
	{
		viSetAttribute (instrSession, VI_ATTR_TERMCHAR, '\n');  	
	}
	
	UNLOCK
}

/*===========================================================================*/
/* Function: IpQuery                                                         */
/* Purpose:  This function reads the terminator attribute.                   */
/*           1 = CR, 2 = LF, 3 = CR LF, 4 = LF CR  							 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_TermQuery (ViSession      instrSession,
                                     ViChar _VI_FAR channelName[],
                                     ViInt32        *terminator)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":NET:TERM?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
	if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *terminator = atof(rdBuffer);

	UNLOCK
}

/*===========================================================================*/
/* Function: ConfigSlaveBaud                                                 */
/* Purpose:  This function sets the Slave Baud Rate to the specified value.  */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ConfigSlaveBaud (ViSession      instrSession,
                                           ViChar _VI_FAR channelName[],
                                           ViReal64        slaveBaud)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK

    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":SLAVEBAUD ");
    sprintf(tmpBuffer,"%f",slaveBaud);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
}

/*===========================================================================*/
/* Function: IpQuery                                                         */
/* Purpose:  This function reads the Slave Baud Rate address.                */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_SlaveBaudQuery (ViSession      instrSession,
                                          ViChar _VI_FAR channelName[],
                                          ViReal64       *slaveBaud)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":SLAVEBAUD?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
	if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *slaveBaud = atof(rdBuffer);

	UNLOCK
}

/*===========================================================================*/
/* Function: ConfigHost                                                        */
/* Purpose:  This function sets the Host information with 					 */
/*           the specified string. 										     */
/*			 Note: 15 character alphanumeric string							 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ConfigHost             (ViSession      instrSession,
                                                  ViChar _VI_FAR channelName[],
                                                  ViChar _VI_FAR SetHost[])
{
  	ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":NET:HOST ");
    strcat(cmdBuffer,SetHost);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: hostQuery                                                         */
/* Purpose:  This function reads the Host information string.                */
/*			 Note: 15 character alphanumeric string							 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_HostQuery              (ViSession      instrSession,
                                                  ViChar _VI_FAR channelName[],
                                                  ViChar _VI_FAR getHost[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViInt16		i;
    ViChar		*cptr;
    ViByte		rdBuffer[BUFFER_SIZE];
    ViByte		cmdBuffer[80];

	LOCK
	
	/* send SCPI identify command */
	strcpy(cmdBuffer,"SYST");
	strcat(cmdBuffer,channelName);
	strcat(cmdBuffer,":NET:HOST");
	strcat(cmdBuffer,"?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
	sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* copy ip string to buffer */
	strcpy(getHost, rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigDesc                                                      */
/* Purpose:  This function sets the desacription information with 		     */
/*           the specified string. 										     */
/*			 Note: 64 character alphanumeric string							 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ConfigDesc             (ViSession      instrSession,
                                                  ViChar _VI_FAR channelName[],
                                                  ViChar _VI_FAR setDesc[])
{
  	ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":NET:DESC ");
    strcat(cmdBuffer,setDesc);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: IpQuery                                                         */
/* Purpose:  This function reads the decription string.                      */
/*			 Note: 64 character alphanumeric string							 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_DescQuery              (ViSession      instrSession,
                                                  ViChar _VI_FAR channelName[],
                                                  ViChar _VI_FAR getDesc[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViInt16		i;
    ViChar		*cptr;
    ViByte		rdBuffer[BUFFER_SIZE];
    ViByte		cmdBuffer[80];

	LOCK
	
	/* send SCPI identify command */
	strcpy(cmdBuffer,"SYST");
	strcat(cmdBuffer,channelName);
	strcat(cmdBuffer,":NET:DESC");
	strcat(cmdBuffer,"?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
	sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* copy ip string to buffer */
	strcpy(getDesc, rdBuffer);

    UNLOCK
}
                                                  
/*===========================================================================*/
/* Function: MACQuery                                                        */
/* Purpose:  This function reads the MAC address.                            */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_MACQuery               (ViSession      instrSession,
                                                  ViChar _VI_FAR channelName[],
                                                  ViChar _VI_FAR getMAC[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViInt16		i;
    ViChar		*cptr;
    ViByte		rdBuffer[BUFFER_SIZE];
    ViByte		cmdBuffer[80];

	LOCK
	
	/* send SCPI identify command */
	strcpy(cmdBuffer,"SYST");
	strcat(cmdBuffer,channelName);
	strcat(cmdBuffer,":NET:MAC");
	strcat(cmdBuffer,"?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
	sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* copy ip string to buffer */
	strcpy(getMAC, rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigAutoIp                                                    */
/* Purpose:  This function sets the Auto IP address mode.                    */
/*           0 = Disable Auto IP, 1 = Enable Auto IP						 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ConfigAutoIP (ViSession      instrSession,
                                        ViChar _VI_FAR channelName[],
                                        ViBoolean      autoIPMode)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":NET:AUTOIP ");
    sprintf(tmpBuffer,"%d",autoIPMode);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
}

/*===========================================================================*/
/* Function: AutoIpQuery                                                     */
/* Purpose:  This function reads the Auto IP address mode.                   */
/*           0 = Disable Auto IP, 1 = Enable Auto IP						 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_AutoIPQuery (ViSession      instrSession,
                                       ViChar _VI_FAR channelName[],
                                       ViBoolean      *autoIPMode)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string - voltage limit */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":NET:AUTOIP?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query string */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *autoIPMode = atoi(rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigDHCPMode                                                  */
/* Purpose:  This function sets the DHCP Mode to the specified value.        */
/*           0 = Static IP address, 1 =  Use DHCP						  	 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ConfigDHCPMode (ViSession      instrSession,
                                          ViChar _VI_FAR channelName[],
                                          ViBoolean      dhcpMode)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":NET:DHCPMODE ");
    sprintf(tmpBuffer,"%d",dhcpMode);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
}

/*===========================================================================*/
/* Function: DHCPModeQuery                                                   */
/* Purpose:  This function reads the DHCp Mode.                              */
/*           0 = Static IP address, 1 =  Use DHCP						  	 */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_DHCPModeQuery (ViSession      instrSession,
                                         ViChar _VI_FAR channelName[],
                                         ViBoolean      *dhcpMode)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string - voltage limit */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":NET:DHCPMODE?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query string */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *dhcpMode = atoi(rdBuffer);

    UNLOCK
}
                                                  
/*===========================================================================*/
/* Function: ConfigLANLED                                                    */
/* Purpose:  This function sets the LANLED Mode to the specified value.      */
/*           1 = BLINKON, 0 = BLINKOFF                    				     */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ConfigLANLED (ViSession      instrSession,
                                        ViChar _VI_FAR channelName[],
                                        ViBoolean      lanLED)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":NET:LANLED ");
    if(lanLED)
    {
    	strcat(cmdBuffer,"BLINKON");   
    }
    else
    {
		strcat(cmdBuffer,"BLINKOFF");       
    }
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
}

/*===========================================================================*/
/* Function: LANLEDQuery                                                     */
/* Purpose:  This function reads the  LANLED Mode.                           */
/*           1 = BLINKON, 0 = BLINKOFF                    				     */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_LANLEDQuery (ViSession      instrSession,
                                       ViChar _VI_FAR channelName[],
                                       ViBoolean      *lanLED)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string - voltage limit */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":NET:LANLED?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query string */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *lanLED = atoi(rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigLastCalDate                                               */
/* Purpose:  This function sets the Last Calibration Date                    */
/*           to the specified value.                                         */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ConfigLastCalDate (ViSession      instrSession,
	                                         ViChar _VI_FAR channelName[],
	                                         ViReal64       month,
	                                         ViReal64       day,
	                                         ViReal64       year)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK

    /* build and send SCPI command string */
    strcpy(cmdBuffer,"CAL");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":MOD:LASTCALDATE ");
    sprintf(tmpBuffer,"%f",month);
    strcat(cmdBuffer,tmpBuffer);
    strcat(cmdBuffer," ");
    sprintf(tmpBuffer,"%f",day);
    strcat(cmdBuffer,tmpBuffer);
    strcat(cmdBuffer," ");
    sprintf(tmpBuffer,"%f",year);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
}

/*===========================================================================*/
/* Function: LastCalDateQuery                                                */
/* Purpose:  This function reads the Last Calibration Date.                  */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_LastCalDateQuery (ViSession      instrSession,
                                            ViChar _VI_FAR channelName[],
                                            ViChar _VI_FAR lastCalDate[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViInt16		i;
    ViChar		*cptr;
    ViByte		rdBuffer[BUFFER_SIZE];
    ViByte		cmdBuffer[80];

	LOCK
	
	/* send SCPI identify command */
	strcpy(cmdBuffer,"CAL");
	strcat(cmdBuffer,channelName);
	strcat(cmdBuffer,":MOD:LASTCALDATE");
	strcat(cmdBuffer,"?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
	sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* copy ip string to buffer */
	strcpy(lastCalDate, rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigNextCalDate                                               */
/* Purpose:  This function sets the Next Calibration Date                    */
/*           to the specified value.                                         */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_ConfigNextCalDate (ViSession      instrSession,
                                             ViChar _VI_FAR channelName[],
                                             ViReal64       month,
                                             ViReal64       day,
                                             ViReal64       year)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK

    /* build and send SCPI command string */
    strcpy(cmdBuffer,"CAL");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":MOD:NEXTCALDATE ");
    sprintf(tmpBuffer,"%f",month);
    strcat(cmdBuffer,tmpBuffer);
    strcat(cmdBuffer," ");
    sprintf(tmpBuffer,"%f",day);
    strcat(cmdBuffer,tmpBuffer);
    strcat(cmdBuffer," ");
    sprintf(tmpBuffer,"%f",year);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
}

/*===========================================================================*/
/* Function: NextCalDateQuery                                                */
/* Purpose:  This function reads the Next Calibration Date.                  */
/*===========================================================================*/
ViStatus CVIFUNC SNSGA_NextCalDateQuery (ViSession      instrSession,
                                            ViChar _VI_FAR channelName[],
                                            ViChar _VI_FAR nextCalDate[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNSGA_status,unlockStatus;
    ViInt16		i;
    ViChar		*cptr;
    ViByte		rdBuffer[BUFFER_SIZE];
    ViByte		cmdBuffer[80];

	LOCK
	
	/* send SCPI identify command */
	strcpy(cmdBuffer,"CAL");
	strcat(cmdBuffer,channelName);
	strcat(cmdBuffer,":MOD:NEXTCALDATE");
	strcat(cmdBuffer,"?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
	sendCnt = strlen(cmdBuffer);
    if ((SNSGA_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
    if ((SNSGA_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* copy ip string to buffer */
	strcpy(nextCalDate, rdBuffer);

    UNLOCK
}


/*****************************************************************************/
/*=== END INSTRUMENT DRIVER SOURCE CODE =====================================*/
/*****************************************************************************/
